const fs = require('fs');
const path = require('path');
const { minify } = require('terser');
const CleanCSS = require('clean-css');

const pluginDir = __dirname;

// JavaScript files to minify
const jsFiles = [
    'js/updates.js',
    'js/modules-admin.js',
    'js/hellopack-client.js'
];

// CSS files to minify
const cssFiles = [
    'css/hellopack-client.css'
];

// Minify JavaScript files
async function minifyJS() {
    console.log('Minifying JavaScript files...');

    for (const file of jsFiles) {
        const inputPath = path.join(pluginDir, file);
        const outputPath = inputPath.replace('.js', '.min.js');

        try {
            const code = fs.readFileSync(inputPath, 'utf8');
            const result = await minify(code, {
                compress: {
                    dead_code: true,
                    drop_console: false,
                    drop_debugger: true,
                    keep_classnames: true,
                    keep_fargs: true,
                    keep_fnames: true,
                    keep_infinity: true
                },
                mangle: false,
                output: {
                    comments: /^!/
                }
            });

            if (result.code) {
                fs.writeFileSync(outputPath, result.code);
                console.log(`✓ ${file} -> ${path.basename(outputPath)}`);
            }
        } catch (error) {
            console.error(`✗ Error minifying ${file}:`, error.message);
        }
    }
}

// Minify CSS files
function minifyCSS() {
    console.log('\nMinifying CSS files...');
    const cleanCSS = new CleanCSS({
        level: 2,
        format: {
            breaks: {
                afterComment: false
            }
        }
    });

    for (const file of cssFiles) {
        const inputPath = path.join(pluginDir, file);
        const outputPath = inputPath.replace('.css', '.min.css');

        try {
            const code = fs.readFileSync(inputPath, 'utf8');
            const result = cleanCSS.minify(code);

            if (result.styles) {
                fs.writeFileSync(outputPath, result.styles);
                console.log(`✓ ${file} -> ${path.basename(outputPath)}`);
            }

            if (result.errors.length > 0) {
                console.error(`✗ Errors in ${file}:`, result.errors);
            }
        } catch (error) {
            console.error(`✗ Error minifying ${file}:`, error.message);
        }
    }
}

// Run minification
async function build() {
    console.log('Starting build process...\n');
    await minifyJS();
    minifyCSS();
    console.log('\n✓ Build complete!');
}

build();
