<?php
/**
 * HelloPack Module Base Class
 *
 * @package HelloPack_Client
 * @since 2.1.0
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Abstract base class for all HelloPack modules.
 */
abstract class HelloPack_Module_Base {

	/**
	 * Module ID
	 *
	 * @var string
	 */
	protected $id = '';

	/**
	 * Module name
	 *
	 * @var string
	 */
	protected $name = '';

	/**
	 * Module description
	 *
	 * @var string
	 */
	protected $description = '';

	/**
	 * Target plugin or theme
	 *
	 * @var array
	 */
	protected $target = array();

	/**
	 * Module category
	 *
	 * @var string
	 */
	protected $category = 'plugin';

	/**
	 * Module version
	 *
	 * @var string
	 */
	protected $version = '1.0.0';

	/**
	 * Module enabled status
	 *
	 * @var bool
	 */
	protected $enabled = true;

	/**
	 * Required plugins
	 *
	 * @var array
	 */
	protected $required_plugins = array();

	/**
	 * Constructor
	 */
	public function __construct() {
		$this->setup();
		
		// Check if module is enabled
		$modules_settings = get_option( 'hellopack_modules_settings', array() );
		if ( isset( $modules_settings[ $this->id ] ) && false === $modules_settings[ $this->id ]['enabled'] ) {
			$this->enabled = false;
		}
	}

	/**
	 * Setup module properties
	 */
	abstract protected function setup();

	/**
	 * Initialize the module
	 */
	abstract public function init();

	/**
	 * Check if module should load
	 *
	 * @return bool
	 */
	public function should_load() {
		// First check if module is enabled in settings
		if ( ! $this->enabled ) {
			return false;
		}

		// Check if module is in disabled list
		if ( defined( 'HP_DISABLE_MODULES' ) ) {
			$disabled = explode( ',', HP_DISABLE_MODULES );
			$disabled = array_map( 'trim', $disabled );
			if ( in_array( $this->id, $disabled, true ) ) {
				return false;
			}
		}

		// Check if target plugin/theme is active
		if ( 'plugin' === $this->category ) {
			if ( ! empty( $this->target['plugin_dir'] ) && ! empty( $this->target['plugin_file'] ) ) {
				return hp_is_plugin_activated( $this->target['plugin_dir'], $this->target['plugin_file'] );
			}
		} elseif ( 'theme' === $this->category ) {
			if ( ! empty( $this->target['theme_slug'] ) ) {
				return hp_is_theme_activated( $this->target['theme_slug'] );
			}
		}

		return false;
	}

	/**
	 * Get module ID
	 *
	 * @return string
	 */
	public function get_id() {
		return $this->id;
	}

	/**
	 * Get module name
	 *
	 * @return string
	 */
	public function get_name() {
		return $this->name;
	}

	/**
	 * Get module description
	 *
	 * @return string
	 */
	public function get_description() {
		return $this->description;
	}

	/**
	 * Get module category
	 *
	 * @return string
	 */
	public function get_category() {
		return $this->category;
	}

	/**
	 * Get module version
	 *
	 * @return string
	 */
	public function get_version() {
		return $this->version;
	}

	/**
	 * Get target info
	 *
	 * @return array
	 */
	public function get_target() {
		return $this->target;
	}

	/**
	 * Check if module is enabled
	 *
	 * @return bool
	 */
	public function is_enabled() {
		return $this->enabled;
	}

	/**
	 * Get module info as array
	 *
	 * @return array
	 */
	public function get_info() {
		$module_loader = HelloPack_Module_Loader::get_instance();
		$is_hard_disabled = false;
		
		// Check if plugin is hard disabled
		if ( 'plugin' === $this->category && ! empty( $this->target['plugin_dir'] ) ) {
			$is_hard_disabled = $module_loader->is_plugin_hard_disabled( $this->target['plugin_dir'] );
		}
		
		return array(
			'id'            => $this->id,
			'name'          => $this->name,
			'description'   => $this->description,
			'category'      => $this->category,
			'version'       => $this->version,
			'target'        => $this->target,
			'enabled'       => $this->enabled,
			'active'        => $this->should_load(),
			'hard_disabled' => $is_hard_disabled,
		);
	}
}