<?php
/**
 * HelloPack Module Loader
 *
 * @package HelloPack_Client
 * @since 2.1.0
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Module loader class.
 */
class HelloPack_Module_Loader {

	/**
	 * Loaded modules
	 *
	 * @var array
	 */
	private $modules = array();

	/**
	 * Module directory
	 *
	 * @var string
	 */
	private $module_dir = '';

	/**
	 * Instance
	 *
	 * @var HelloPack_Module_Loader
	 */
	private static $instance = null;

	/**
	 * Get instance
	 *
	 * @return HelloPack_Module_Loader
	 */
	public static function get_instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Constructor
	 */
	private function __construct() {
		$this->module_dir = HELLOPACK_CLIENT_PATH . 'inc/modules/';
		
		// Create modules directory if it doesn't exist
		if ( ! is_dir( $this->module_dir ) ) {
			wp_mkdir_p( $this->module_dir );
		}

		// Load module base class
		require_once HELLOPACK_CLIENT_PATH . 'inc/class-hellopack-module-base.php';
		
		// Initialize modules
		add_action( 'init', array( $this, 'load_modules' ), 5 );
	}

	/**
	 * Load all modules
	 */
	public function load_modules() {
		// Get all PHP files from modules directory
		$module_files = glob( $this->module_dir . '*.php' );
		
		if ( ! empty( $module_files ) ) {
			foreach ( $module_files as $file ) {
				$this->load_module( $file );
			}
		}

		// Also load legacy modules if they exist
		$this->load_legacy_modules();

		// Allow other plugins to register modules
		do_action( 'hellopack_register_modules', $this );
	}

	/**
	 * Load a single module
	 *
	 * @param string $file Module file path.
	 */
	private function load_module( $file ) {
		if ( ! file_exists( $file ) ) {
			return;
		}

		// Include the file
		require_once $file;

		// Get class name from file name
		$filename = basename( $file, '.php' );
		$class_name = $this->get_class_name_from_file( $filename );

		// Check if class exists and is a subclass of HelloPack_Module_Base
		if ( class_exists( $class_name ) && is_subclass_of( $class_name, 'HelloPack_Module_Base' ) ) {
			$module = new $class_name();

			// Register module if its target is active OR if it's hard-disabled (so it shows in admin)
			// OR if it's a theme module (themes should always show in admin)
			if ( $this->is_module_target_active( $module ) || $this->is_module_hard_disabled( $module ) || 'theme' === $module->get_category() ) {
				$this->register_module( $module );
			}
		}
	}

	/**
	 * Load legacy modules with new wrapper
	 */
	private function load_legacy_modules() {
		$legacy_dir = HELLOPACK_CLIENT_PATH . 'inc/legacy/modules/';
		
		if ( ! is_dir( $legacy_dir ) ) {
			return;
		}

		$legacy_files = glob( $legacy_dir . 'module-*.php' );
		
		if ( ! empty( $legacy_files ) ) {
			foreach ( $legacy_files as $file ) {
				// For now, just include them as before
				// Later we can wrap them in the new module system
				if ( file_exists( $file ) ) {
					include_once $file;
				}
			}
		}
	}

	/**
	 * Get class name from file name
	 *
	 * @param string $filename File name without extension.
	 * @return string
	 */
	private function get_class_name_from_file( $filename ) {
		// Convert filename to class name
		// e.g., 'hellopack-module-acf-pro' becomes 'HelloPack_Module_ACF_Pro'
		$parts = explode( '-', $filename );
		$parts = array_map( 'ucfirst', $parts );

		// Special handling for common abbreviations and compound words
		$replacements = array(
			'Hellopack' => 'HelloPack',
			'Acf' => 'ACF',
			'Wp' => 'WP',
			'Seo' => 'SEO',
		);

		foreach ( $replacements as $search => $replace ) {
			$parts = array_map( function( $part ) use ( $search, $replace ) {
				return $part === $search ? $replace : $part;
			}, $parts );
		}

		return implode( '_', $parts );
	}

	/**
	 * Register a module
	 *
	 * @param HelloPack_Module_Base $module Module instance.
	 */
	public function register_module( HelloPack_Module_Base $module ) {
		$module_id = $module->get_id();
		
		// Store module instance
		$this->modules[ $module_id ] = $module;
		
		// Initialize module if it should load
		if ( $module->should_load() ) {
			$module->init();
		}
	}

	/**
	 * Get all registered modules
	 *
	 * @return array
	 */
	public function get_modules() {
		return $this->modules;
	}

	/**
	 * Get module by ID
	 *
	 * @param string $module_id Module ID.
	 * @return HelloPack_Module_Base|null
	 */
	public function get_module( $module_id ) {
		return isset( $this->modules[ $module_id ] ) ? $this->modules[ $module_id ] : null;
	}

	/**
	 * Get modules by category
	 *
	 * @param string $category Category name.
	 * @return array
	 */
	public function get_modules_by_category( $category ) {
		$filtered = array();
		
		foreach ( $this->modules as $id => $module ) {
			if ( $module->get_category() === $category ) {
				$filtered[ $id ] = $module;
			}
		}
		
		return $filtered;
	}

	/**
	 * Update module settings
	 *
	 * @param array $settings Module settings.
	 */
	public function update_module_settings( $settings ) {
		update_option( 'hellopack_modules_settings', $settings );
		
		// Reload modules to apply new settings
		$this->modules = array();
		$this->load_modules();
	}

	/**
	 * Get module settings
	 *
	 * @return array
	 */
	public function get_module_settings() {
		return get_option( 'hellopack_modules_settings', array() );
	}

	/**
	 * Check if module target is active
	 *
	 * @param HelloPack_Module_Base $module Module instance.
	 * @return bool
	 */
	private function is_module_target_active( HelloPack_Module_Base $module ) {
		$target = $module->get_target();
		$category = $module->get_category();

		if ( 'plugin' === $category ) {
			if ( ! empty( $target['plugin_dir'] ) && ! empty( $target['plugin_file'] ) ) {
				return hp_is_plugin_activated( $target['plugin_dir'], $target['plugin_file'] );
			}
		} elseif ( 'theme' === $category ) {
			if ( ! empty( $target['theme_slug'] ) ) {
				return hp_is_theme_activated( $target['theme_slug'] );
			}
		}

		return false;
	}

	/**
	 * Check if plugin is hard disabled via HP_DISABLE_PLUGINS constant
	 *
	 * @param string $plugin_dir Plugin directory name.
	 * @return bool
	 */
	public function is_plugin_hard_disabled( $plugin_dir ) {
		if ( ! defined( 'HP_DISABLE_PLUGINS' ) ) {
			return false;
		}

		$disabled_plugins = explode( ',', HP_DISABLE_PLUGINS );
		$disabled_plugins = array_map( 'trim', $disabled_plugins );

		return in_array( $plugin_dir, $disabled_plugins, true );
	}

	/**
	 * Check if module is hard disabled
	 *
	 * @param HelloPack_Module_Base $module Module instance.
	 * @return bool
	 */
	private function is_module_hard_disabled( HelloPack_Module_Base $module ) {
		$target = $module->get_target();
		$category = $module->get_category();

		if ( 'plugin' === $category && ! empty( $target['plugin_dir'] ) ) {
			return $this->is_plugin_hard_disabled( $target['plugin_dir'] );
		}

		return false;
	}
}