<?php
/**
 * HelloPack Module: WPML
 *
 * @package HelloPack
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

/**
 * Class HelloPack_Module_WPML
 *
 * Module for managing WPML plugin functionality
 */
class HelloPack_Module_WPML extends HelloPack_Module_Base {

	/**
	 * Module setup
	 *
	 * @return void
	 */
	protected function setup() {
		$this->id          = 'wpml';
		$this->name        = 'WPML';
		$this->description = __( 'Manages WPML updates through centralized updater.', 'hellopack-client' );
		$this->category    = 'plugin';
		$this->version     = '1.0.0';
		$this->target      = array(
			'plugin_dir'  => 'sitepress-multilingual-cms',
			'plugin_file' => 'sitepress.php',
		);
	}

	/**
	 * Module initialization
	 *
	 * @return void
	 */
	public function init() {
		if ( ! $this->should_load() ) {
			return;
		}

		// Set up API servers
		$wpml = new HPack_Set_API_Servers();
		$wpml->set_api_servers( 'api.wpml.org', HP_PLUGIN_REGISTER_SERVER . '/wpml' );
		$wpml->init();

		$toolset = new HPack_Block_API_Servers();
		$toolset->set_api_servers( 'toolset.com' );
		$toolset->init();

		$cdnwpml = new HPack_Block_API_Servers();
		$cdnwpml->set_api_servers( 'cdn.wpml.org' );
		$cdnwpml->init();

		// Check and update WPML settings
		$this->update_wpml_installer_settings();

		// Add action hooks
		add_action( 'wp_loaded', array( $this, 'wpml_dont_overwrite_the_wp_pls' ) );
		add_action( 'init', array( $this, 'update_wpml_setup' ), 9999 );
	}

	/**
	 * Update WPML installer settings
	 *
	 * @return void
	 */
	private function update_wpml_installer_settings() {
		if ( get_option( 'wp_installer_settings' ) !== get_option( 'wp_installer_settings_backup' ) ) {
			$json_url = HP_PLUGIN_REGISTER_SERVER_HTTPS . '/wpml/set/';
			$response = wp_remote_get( $json_url );

			if ( is_array( $response ) && ! is_wp_error( $response ) ) {
				$json_data       = wp_remote_retrieve_body( $response );
				$json_data_array = json_decode( $json_data, true );
				$json_data_array = $json_data_array['api-data'];

				$wp_installer_settings = get_option( 'wp_installer_settings' );

				if ( $wp_installer_settings ) {
					update_option( 'wp_installer_settings', $json_data_array );
					update_option( 'wp_installer_settings_backup', $json_data_array );
				} else {
					add_option( 'wp_installer_settings', $json_data_array );
					add_option( 'wp_installer_settings_backup', $json_data_array );
				}
			}
		}
	}

	/**
	 * Prevent WPML from overwriting WP actions
	 *
	 * @return void
	 */
	public function wpml_dont_overwrite_the_wp_pls() {
		// Only run once
		if ( did_action( 'wpml_dont_overwrite_the_wp_pls_executed' ) ) {
			return;
		}

		try {
			remove_all_actions( 'wp_ajax_update-plugin' );
		} catch ( Exception $e ) {
			// Silently fail
		}

		// Mark as executed
		do_action( 'wpml_dont_overwrite_the_wp_pls_executed' );
	}

	/**
	 * Update WPML setup options
	 *
	 * @return void
	 */
	public function update_wpml_setup() {
		$wpml_settings_exist = get_option( 'WPML(setup)', false );

		if ( false === $wpml_settings_exist ) {
			$wpml_settings_exist                  = array();
			$wpml_settings_exist['is-tm-allowed'] = true;
			add_option( 'WPML(setup)', $wpml_settings_exist );
			return;
		}

		$wpml_settings = get_option( 'WPML(setup)' );
		if ( ! is_array( $wpml_settings ) ) {
			$wpml_settings = array();
		}

		// Only update if necessary
		if ( ! isset( $wpml_settings['is-tm-allowed'] ) || $wpml_settings['is-tm-allowed'] !== true ) {
			$wpml_settings['is-tm-allowed'] = true;
			update_option( 'WPML(setup)', $wpml_settings );
		}
	}
}